/** @file   aicontroller.h
 * @brief   Declarations of AIController - class.
 * @version $Revision: 1.3 $
 * @date    $Date: 2006/03/16 21:00:28 $
 * @author  Tomi Lamminsaari
 */
 
#ifndef AICONTROLLER_H
#define AICONTROLLER_H

#include "basecontroller.h"

namespace WeWantWar {


/** @class  AIController
 * @brief   Extends the BaseController with some AI interface.
 * @author  Tomi Lamminsaari
 *
 * This class provides some utility methods that can be useful when implementing
 * actual behaviour logic. These methods provides random movement, calculates
 * the line-of-sights, avoid collisions to obstacles.
 *
 * The actual concrete controller class must implement the
 * <code>void updateAI(bool seeTarget, float dist)</code> - method where the
 * actual behaviour shall be implemented.
 *
 * The @c updateAI(...) method should modify the members ( such as @c m_forward )
 * in @c BaseController to achieve desired behaviour.
 */
class AIController : public BaseController, public eng2d::MMapDataProvider
{
public:

  ///
  /// Constants
  /// =========

  /** A datatype for the behaviour flags. These flags are more or less
   * suggestive because the actual ai-logic decides wether to respect
   * these flags or not.
   */
  typedef unsigned int BFlag;
  
  /** Indicates that we should stay put until we see the target. */
  static const BFlag WAIT_TARGET = 1;
  /** Indicates that we should try to find the player if he hides from us. */
  static const BFlag CHASE_TARGET = 1 << 1;
  /** Indicates that we should not move at all. */
  static const BFlag STAY_PUT = 1 << 2;
  
  /** THe possible modes the Ai controller can operate.
   */
  enum AiMode {
    EGuarding,
    EAlerted,
    EFrozen
  };
  
  ///
  /// Constructors, destructor
  /// ========================
  
  /** Constructor
   * @param     pAlien            Pointer to GameObject we're supposed to control
   */
  AIController( GameObject* pAlien );
  
  /** Constructor.
   * @param     pAlien            Pointer to GameObject we're controlling.
   * @param     pTarget           Pointer to our target GameObject
   */
  AIController( GameObject* pAlien, GameObject* pTarget );
  
  /** Destructor.
   */
  virtual ~AIController();
  
  
  
  ///
  /// Methods
  /// =======

  /** From MMapDataProvider
   */
  virtual float ApproximateDistance( const eng2d::Vec2D& aPoint1,
                                     const eng2d::Vec2D& aPoint2 ) const;

  /** From MMapDataProvider
   */
  virtual bool IsSameNode( const eng2d::Vec2D& aPoint1,
                           const eng2d::Vec2D& aPoint2 ) const;

  /** From MMapDataProvider
   */
  virtual void GetAdjacentNodes( std::vector<eng2d::Vec2D>& aAdjacentNodes,
                                 const eng2d::Vec2D& aCurrentPos ) const;

  /** From MMapDataProvider
   */
  virtual float GetTerrainFactor( const eng2d::Vec2D& aPos ) const;
 
  
  /** Sets the target object.
   * @param     pTarget           Pointer to GameObject that is our target.
   */
  virtual void setTarget( GameObject* pTarget );
  
  
  /** Sets the "behaviour" flags.
   * @param     f                 A bitfield that contains the behaviour
   *                              flags.
   */
  virtual void setFlags( BFlag f );

  /** The update method.
   */
  virtual void update();
  
  /** This method should take care updating the actual controller. This gets
   * called from the @c update() - method.
   * @param     seeTarget         Do we see the target.
   * @param     dist              Distance from target.
   */
  virtual void updateAI( bool seeTarget, float dist ) = 0;
  
  /** Sets the distance how far this controller can see.
   * @param   aDistance         New seeing distance.
   */
  void setVisualRange( float aDistance );

  /** Sets the attacking range.
   * @param   aDistance         The attackng distance
   */
  void setAttackDistance( float aDistance );
  
  
  ///
  /// Getter methods
  /// ==============
  
  /** Returns the target object.
   * @return    Pointer to target object.
   */
  inline GameObject* getTarget() const { return m_pTarget; }
  
  /** Returns the flags
   * @return    The bitfield containing the behaviour flags.
   */
  BFlag getFlags() const;
  
  
protected:

  /** Checks if alien we're controlling can see the target.
   * @return    'true' if alien can see player.
   */
  virtual bool canSeeTarget() const;


  /** Tries to keep the alien away from the walls.
   */
  virtual void avoidWalls();

  /** Sets the random movement data.
   * @param     counter           How long these controls are valid.
   */
  virtual void setRandomMovement( int counter );

  /** Tells if the object is within the attacking range from the target
   * @return    <code>true</code> if we're within the attacking range.
   */
  virtual bool withinAttackRange() const;

  /** Tells if we're within the visual range.
   * @return    <code>true</code> if we're within the visual range.
   */
  virtual bool withinVisualRange() const;


  ///
  /// Members
  /// =======
  
  /** Pointer to our target object. */
  GameObject*   m_pTarget;
  
  /** The maximum distance this controller can react with its target. */
  float m_visualRange;

  /** The attacking range */
  float m_attackRange;

  /** A flag that indicates have we seen our target yet. */
  bool m_targetSeen;
  
  /** This bitfield contains the behaviour flags. */
  BFlag  m_flags;
  
  /** An update-counter we use when trying to prevent the object getting
   * stuck.
   */
  int m_updateCounter;
  
  /** The position we were 10 updates ago. */
  eng2d::Vec2D  m_oldPos;

  /** Indicates when we're stuck */
  bool m_stuck;

  
private:
  /** Disabled copy constructor. */
  AIController( const AIController& rAIC );
  /** Disabled assignment operator */
  AIController& operator = ( const AIController& rAIC );
};

};  // end of namespace

#endif // AICONTROLLER_H

/**
 * Version history
 * ===============
 * $Log: aicontroller.h,v $
 * Revision 1.3  2006/03/16 21:00:28  lamminsa
 * setVisualRange and setAttackDistance() methods added to gain
 * possibility to change the ai object's behaviour anytime.
 *
 * Revision 1.2  2006/02/26 07:39:22  lamminsa
 * no message
 *
 * Revision 1.1.1.1  2006/01/21 23:02:41  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:16:46+02  lamminsa
 * Initial revision
 *
 */
